<?php
/*******************************************************
 * TradeJourney – Trading Journal (manual)
 * Password protection (user/pass & CSV path via config_journey.php)
 * CSV: data_journey.csv (configurable)
 * Time format: YYYY-MM-DD HH:MM (no seconds)
 *******************************************************/
declare(strict_types=1);
session_start();

/* ====== Load configuration (config_journey.php) ====== */
// Adjust the path if you keep the config outside the webroot:
$config = require __DIR__ . '/config_journey.php';
include 'analytics.php';

$CONF_USER = $config['user']      ?? 'admin';
$CONF_HASH = $config['pass_hash'] ?? '';                 // optional
$CONF_PASS = $config['pass']      ?? '';                 // fallback only
$CSV_FILE  = $config['csv']       ?? (__DIR__ . '/data_journey.csv');

// If CSV path is relative, resolve it against this file:
if ($CSV_FILE !== '' && $CSV_FILE[0] !== '/' && !preg_match('~^[A-Za-z]:\\\\~', $CSV_FILE)) {
  $CSV_FILE = __DIR__ . '/' . ltrim($CSV_FILE, '/');
}

/* ====== Login/Logout handling ====== */
if (isset($_GET['logout'])) {
  $_SESSION = [];
  if (ini_get("session.use_cookies")) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time()-42000, $params["path"], $params["domain"], $params["secure"], $params["httponly"]);
  }
  session_destroy();
  header('Location: '.$_SERVER['PHP_SELF']);
  exit;
}

$login_error = '';
if (($_SERVER['REQUEST_METHOD']==='POST') && isset($_POST['do_login'])) {
  $u = trim($_POST['user'] ?? '');
  $p = (string)($_POST['pass'] ?? '');
  $ok = false;

  if (!empty($CONF_HASH)) {
    // Prefer hashed password
    $ok = ($u === $CONF_USER) && password_verify($p, $CONF_HASH);
  } else {
    // Fallback: plaintext
    $ok = ($u === $CONF_USER) && ($p === $CONF_PASS);
  }

  if ($ok) {
    $_SESSION['tj_auth'] = true;
    header('Location: '.$_SERVER['PHP_SELF']);
    exit;
  } else {
    $login_error = 'Invalid credentials.';
  }
}

/* ====== Unauth: Login page ====== */
if (!isset($_SESSION['tj_auth']) || $_SESSION['tj_auth'] !== true) {
  ?>
  <!doctype html>
  <html lang="en">
  <head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>TradeJourney – Login</title>
    <style>
      :root{--bg:#f7fafc;--card:#fff;--text:#0f172a;--muted:#64748b;--border:#e5e7eb;--acc:#2563eb;--bad:#dc2626}
      *{box-sizing:border-box}
      body{margin:0;background:#d1edfd;color:var(--text);font-family:system-ui,-apple-system,Segoe UI,Roboto}
      .wrap{min-height:100vh;display:grid;place-items:center;padding:16px}
      .card{width:min(460px,92vw);background:var(--card);border:1px solid var(--border);border-radius:16px;padding:20px;box-shadow:0 10px 30px rgba(0,0,0,.06)}
      .brand{display:flex;align-items:center;gap:16px;margin-bottom:10px}
      .brand img{height:54px}
      h1{margin:0;font-size:1.6rem}
      label{display:block;margin:.6rem 0 .3rem;font-weight:700}
      input{width:100%;padding:12px;border:1px solid var(--border);border-radius:12px}
      .btn{margin-top:14px;width:100%;padding:12px;border:0;border-radius:12px;background:var(--acc);color:#fff;font-weight:700;cursor:pointer}
      .err{color:var(--bad);margin-top:8px}
      .muted{color:var(--muted);font-size:.9rem;margin-top:8px}
      code{background:#f1f5f9;padding:2px 6px;border-radius:6px}
    </style>
  </head>
  <body>
    <div class="wrap">
      <div class="card">
        <div class="brand">
          <img src="https://ea-marketplace.com/images/market-trader-logo.webp" alt="Market Trader">
          <div>
            <h1>TradeJourney</h1>
            <div class="muted">Please sign in</div>
          </div>
        </div>
        <form method="post" autocomplete="off">
          <input type="hidden" name="do_login" value="1">
          <label>Username</label>
          <input name="user" required>
          <label>Password</label>
          <input type="password" name="pass" required>
          <button class="btn" type="submit">Sign in</button>
          <?php if($login_error){ echo '<div class="err">'.htmlspecialchars($login_error,ENT_QUOTES).'</div>'; } ?>
          <div class="muted"></div>
        </form>
      </div>
    </div>
  </body>
  </html>
  <?php
  exit;
}

/* ====== App (authenticated) ====== */

/* ---------- CSV header ---------- */
$HEAD = [
  'ts_open','ts_close','symbol','direction','entry','sl','tp','exit',
  'rr','setup','comment','volume','profit_eur'
];

/* ---------- First run: create sample CSV ---------- */
if (!is_file($CSV_FILE)) {
  $sample = [
    $HEAD,
    ['2025-09-22 08:59','2025-09-22 10:17','DE40','Long','15890','15845','15980','15955','1.44','ORB','Opening range','1','65.00'],
    ['2025-09-22 13:05','2025-09-22 14:02','DE40','Short','16010','16060','15920','15940','1.40','Pullback','Trend break','1','70.00'],
    ['2025-09-23 09:02','2025-09-23 11:48','XAUUSD','Long','2445.20','2441.50','2452.00','2442.10','-0.82','ATR','stop run','0.5','-155.00'],
    ['2025-09-25 09:01','2025-09-25 10:36','DE40','Long','15910','15870','16020','16005','2.38','ORB','','1','238.00'],
  ];
  if ($fh = @fopen($CSV_FILE,'w')) {
    foreach($sample as $row){ fputcsv($fh,$row,';'); }
    fclose($fh);
  } else {
    error_log("TradeJourney: could not create CSV: ".$CSV_FILE);
  }
}

/* ---------- Helpers ---------- */
function csv_read_assoc(string $file): array {
  if (!is_file($file)) return [[],[]];
  $fh=fopen($file,'r'); if(!$fh) return [[],[]];
  $head=fgetcsv($fh,0,';'); if(!$head){ fclose($fh); return [[],[]]; }
  $rows=[];
  while(($r=fgetcsv($fh,0,';'))!==false){
    if(count($r)<count($head)) continue;
    $rows[] = array_combine($head,$r);
  }
  fclose($fh);
  return [$head,$rows];
}
function csv_write_all(string $file, array $head, array $rows): bool {
  $fh=fopen($file,'w'); if(!$fh) return false;
  fputcsv($fh,$head,';');
  foreach($rows as $r){
    $out=[]; foreach($head as $k){ $out[]=$r[$k]??''; }
    fputcsv($fh,$out,';');
  }
  fclose($fh); return true;
}
function csv_append_row(string $file, array $head, array $row): bool {
  $fh=fopen($file,'a'); if(!$fh) return false;
  $out=[]; foreach($head as $k){ $out[]=$row[$k]??''; }
  $ok = fputcsv($fh,$out,';')!==false; fclose($fh); return $ok;
}
function s($v,$def=''){ return htmlspecialchars((string)($v??$def), ENT_QUOTES, 'UTF-8'); }
function num($v): float { return is_numeric($v)? (float)$v : 0.0; }
function fmt($v,$d=2){ return number_format((float)$v,$d,',','.'); }
function dt_local_to_str(?string $dl): string { return $dl ? str_replace('T',' ',$dl) : ''; }
function dt_str_to_local(?string $s): string { return $s ? str_replace(' ','T',$s) : ''; }
function valid_dt_min(?string $s): bool { return $s ? (bool)preg_match('~^\d{4}-\d{2}-\d{2}\s\d{2}:\d{2}$~',$s) : false; }

/* ---------- R multiple in price units ---------- */
function compute_r_points(array $in): float {
  $dir   = $in['direction'] ?? 'Long';
  $entry = (float)str_replace(',','.',$in['entry'] ?? '');
  $exit  = (float)str_replace(',','.',$in['exit']  ?? '');
  $sl    = (float)str_replace(',','.',$in['sl']    ?? '');
  if($entry<=0 || $sl<=0 || $exit<=0) return 0.0;
  if($dir==='Long' && $entry>$sl){
    $risk = $entry - $sl; $gain = $exit - $entry; return ($risk>0)? round($gain/$risk,2) : 0.0;
  }
  if($dir==='Short' && $sl>$entry){
    $risk = $sl - $entry; $gain = $entry - $exit; return ($risk>0)? round($gain/$risk,2) : 0.0;
  }
  return 0.0;
}

/* ---------- Load data ---------- */
[$HEAD_LOADED,$DATA] = csv_read_assoc($CSV_FILE);
if (!$HEAD_LOADED) { $HEAD_LOADED = $HEAD; }

/* ---------- Actions ---------- */
$msg = '';
if ($_SERVER['REQUEST_METHOD']==='POST') {
  $act = $_POST['act'] ?? '';
  if ($act === 'add') {
    $row = [
      'ts_open'    => dt_local_to_str($_POST['ts_open']  ?? ''),
      'ts_close'   => dt_local_to_str($_POST['ts_close'] ?? ''),
      'symbol'     => strtoupper(trim($_POST['symbol'] ?? '')),
      'direction'  => $_POST['direction'] ?? 'Long',
      'entry'      => str_replace(',','.', $_POST['entry'] ?? ''),
      'sl'         => str_replace(',','.', $_POST['sl'] ?? ''),
      'tp'         => str_replace(',','.', $_POST['tp'] ?? ''),
      'exit'       => str_replace(',','.', $_POST['exit'] ?? ''),
      'rr'         => '0.00',
      'setup'      => trim($_POST['setup'] ?? ''),
      'comment'    => trim($_POST['comment'] ?? ''),
      'volume'     => str_replace(',','.', $_POST['volume'] ?? '1'),
      'profit_eur' => str_replace(',','.', $_POST['profit_eur'] ?? ''),
    ];
    if (!$row['symbol'] || !in_array($row['direction'],['Long','Short'],true) || !valid_dt_min($row['ts_close'])) {
      $msg = 'Please provide symbol, direction and a valid close time (YYYY-MM-DD HH:MM).';
    } else {
      $row['rr'] = sprintf('%.2f', compute_r_points($row));
      csv_append_row($CSV_FILE, $HEAD_LOADED, $row);
      header('Location: '.$_SERVER['PHP_SELF'].'?tab=stats');
      exit;
    }
  } elseif ($act === 'edit_save') {
    $idx = (int)($_POST['idx'] ?? -1);
    if ($idx>=0 && $idx < count($DATA)) {
      $row = [
        'ts_open'    => dt_local_to_str($_POST['ts_open']  ?? ''),
        'ts_close'   => dt_local_to_str($_POST['ts_close'] ?? ''),
        'symbol'     => strtoupper(trim($_POST['symbol'] ?? '')),
        'direction'  => $_POST['direction'] ?? 'Long',
        'entry'      => str_replace(',','.', $_POST['entry'] ?? ''),
        'sl'         => str_replace(',','.', $_POST['sl'] ?? ''),
        'tp'         => str_replace(',','.', $_POST['tp'] ?? ''),
        'exit'       => str_replace(',','.', $_POST['exit'] ?? ''),
        'rr'         => '0.00',
        'setup'      => trim($_POST['setup'] ?? ''),
        'comment'    => trim($_POST['comment'] ?? ''),
        'volume'     => str_replace(',','.', $_POST['volume'] ?? '1'),
        'profit_eur' => str_replace(',','.', $_POST['profit_eur'] ?? ''),
      ];
      $row['rr'] = sprintf('%.2f', compute_r_points($row));
      $DATA[$idx] = $row;
      csv_write_all($CSV_FILE, $HEAD_LOADED, $DATA);
    }
    echo '<!doctype html><meta charset="utf-8"><script>if(window.opener){window.opener.location.reload();}window.close();</script>';
    exit;
  }
}

/* ---------- Popup editor ---------- */
if (isset($_GET['edit'])) {
  $idx = (int)$_GET['edit'];
  if ($idx<0 || $idx>=count($DATA)) { echo 'Invalid index'; exit; }
  $r = $DATA[$idx];
  ?>
  <!doctype html>
  <html lang="en">
  <head>
    <meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
    <title>Edit trade</title>
    <style>
      *{box-sizing:border-box}
      body{font-family:system-ui,Segoe UI,Roboto;margin:16px}
      label{display:block;margin:6px 0 4px;color:#111;font-weight:700}
      input,select,textarea{width:100%;padding:10px;border:1px solid #ddd;border-radius:10px}
      .grid{display:grid;grid-template-columns:1fr 1fr;column-gap:16px;row-gap:12px}
      .grid > div{min-width:0}
      .btns{margin-top:14px;display:flex;gap:10px}
      button{padding:10px 14px;border:0;border-radius:10px;background:#2563eb;color:#fff;cursor:pointer}
      @media (max-width: 720px){ .grid{grid-template-columns:1fr} }
    </style>
  </head>
  <body>
    <h3>Edit trade</h3>
    <form method="post">
      <input type="hidden" name="act" value="edit_save">
      <input type="hidden" name="idx" value="<?php echo $idx; ?>">
      <div class="grid">
        <div><label>Open time</label><input type="datetime-local" name="ts_open" step="60" value="<?php echo s(dt_str_to_local($r['ts_open'])); ?>"></div>
        <div><label>Close time</label><input type="datetime-local" name="ts_close" step="60" value="<?php echo s(dt_str_to_local($r['ts_close'])); ?>" required></div>
        <div><label>Symbol</label><input name="symbol" value="<?php echo s($r['symbol']); ?>" required></div>
        <div><label>Direction</label>
          <select name="direction">
            <option <?php echo ($r['direction']==='Long'?'selected':''); ?>>Long</option>
            <option <?php echo ($r['direction']==='Short'?'selected':''); ?>>Short</option>
          </select>
        </div>
        <div><label>Entry</label><input name="entry" value="<?php echo s($r['entry']); ?>"></div>
        <div><label>Exit</label><input name="exit" value="<?php echo s($r['exit']); ?>"></div>
        <div><label>SL</label><input name="sl" value="<?php echo s($r['sl']); ?>"></div>
        <div><label>TP</label><input name="tp" value="<?php echo s($r['tp']); ?>"></div>
        <div><label>Volume</label><input name="volume" value="<?php echo s($r['volume']); ?>"></div>
        <div><label>Profit (€)</label><input name="profit_eur" value="<?php echo s($r['profit_eur']); ?>"></div>
        <div style="grid-column:1/-1"><label>Setup</label><input name="setup" value="<?php echo s($r['setup']); ?>"></div>
        <div style="grid-column:1/-1"><label>Notes</label><textarea name="comment" rows="2"><?php echo s($r['comment']); ?></textarea></div>
      </div>
      <div class="btns">
        <button type="submit">OK (Save)</button>
        <button type="button" onclick="window.close()">Cancel</button>
      </div>
    </form>
  </body>
  </html>
  <?php
  exit;
}

/* ---------- Stats & aggregations ---------- */
$tab = $_GET['tab'] ?? 'input';

$total = count($DATA);
$wins=0; $loss=0; $brut_win=0.0; $brut_loss=0.0;
foreach($DATA as $r){
  $p=num($r['profit_eur']??0);
  if ($p>0){ $wins++;  $brut_win += $p; }
  elseif ($p<0){ $loss++; $brut_loss += -$p; }
}
$net = $brut_win - $brut_loss;
$pf  = $brut_loss>0 ? $brut_win/$brut_loss : ($brut_win>0?INF:0);
$hit = $total>0 ? $wins/$total*100 : 0;
$avg = $total>0 ? $net/$total : 0;

/* Equity */
$DATA_SORT = $DATA;
usort($DATA_SORT, function($a,$b){ return strcmp($a['ts_close'],$b['ts_close']); });
$labels=[]; $equity=[]; $eq=0.0;
foreach($DATA_SORT as $r){ $eq += num($r['profit_eur']); $equity[] = round($eq,2); $labels[] = $r['ts_close']; }

/* Wins/Loss per symbol */
$wl=[];
foreach($DATA as $r){
  $s=$r['symbol']??''; if(!$s) continue;
  if(!isset($wl[$s])) $wl[$s]=['w'=>0,'l'=>0];
  $p=num($r['profit_eur']);
  if($p>0) $wl[$s]['w']++; elseif($p<0) $wl[$s]['l']++;
}
$wl_labels=array_keys($wl);
$wl_w=array_map(fn($x)=>$x['w'],$wl);
$wl_l=array_map(fn($x)=>$x['l'],$wl);

/* Long vs Short PnL */
$sumL = array_sum(array_map(fn($r)=>($r['direction']==='Long')? num($r['profit_eur']) : 0,$DATA));
$sumS = array_sum(array_map(fn($r)=>($r['direction']==='Short')?num($r['profit_eur']) : 0,$DATA));
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>TradeJourney – Trading Journal</title>
<link rel="preconnect" href="https://cdn.jsdelivr.net" crossorigin>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<style>
:root{--bg:#f7fafc;--text:#0f172a;--muted:#6b7280;--card:#fff;--border:#e5e7eb;--ok:#16a34a;--bad:#dc2626;--acc:#2563eb}
*{box-sizing:border-box} body{margin:0;background:#f8fafc;color:var(--text);font-family:system-ui,-apple-system,Segoe UI,Roboto}
.wrap{max-width:75%;margin:28px auto;padding:0 16px}
.topbar{display:flex;align-items:center;justify-content:space-between;margin-bottom:10px}
.header{display:flex;align-items:center;gap:32px}
.header img{height:65px;width:auto;display:block}
.brand{display:flex;flex-direction:column;gap:2px}
.brand .title{margin:0;font-size:1.8rem;line-height:1.1;font-weight:800;letter-spacing:.2px}
.brand .tag{color:var(--muted);font-size:.9rem}
.logout{display:inline-flex;align-items:center;gap:8px;padding:10px 14px;border:1px solid var(--border);border-radius:12px;background:#fff;text-decoration:none;color:#0f172a}
.logout:hover{box-shadow:0 6px 16px rgba(0,0,0,.06)}
.tabs{display:flex;gap:8px;margin:12px 0}
.tab{padding:8px 12px;border:1px solid var(--border);background:var(--card);border-radius:10px;text-decoration:none;color:var(--text)}
.tab.active{border-color:#cbd5e1;box-shadow:0 6px 18px rgba(0,0,0,.05)}
.grid{display:grid;gap:12px;grid-template-columns:repeat(auto-fit,minmax(260px,1fr))}
.card{background:var(--card);border:1px solid var(--border);border-radius:14px;padding:14px;box-shadow:0 8px 20px rgba(0,0,0,.05)}
.card form label{display:block;font-size:.9rem;color:#111;margin-bottom:6px;font-weight:700}
input,select,textarea{width:100%;padding:10px;border:1px solid var(--border);border-radius:10px}
.btn{background:var(--acc);color:#fff;border:none;padding:10px 14px;border-radius:10px;cursor:pointer;font-weight:600}
.kpi{display:flex;justify-content:space-between;margin:6px 0}.kpi .lbl{color:var(--muted)}.kpi .val{font-weight:700}
table{width:100%;border-collapse:collapse;margin-top:8px} th,td{padding:8px 8px;border-bottom:1px solid var(--border);text-align:left}
tr.clickable{cursor:pointer}
.ok{color:var(--ok)} .bad{color:var(--bad)} .acc{color:var(--acc)} .small{color:var(--muted);font-size:.85rem}
.metrics{display:grid;gap:12px;grid-template-columns:repeat(auto-fit,minmax(240px,1fr))}
.metric{background:var(--card);border:1px solid var(--border);border-radius:14px;padding:14px}
.metric h4{margin:0 0 6px 0;font-size:1rem;color:#111}
.metric .big{font-size:1.4rem;font-weight:800}
.metric .sub{color:var(--muted);font-size:.9rem}
/* wrap long paths nicely */
.metric .sub{overflow-wrap:anywhere; word-break:break-word}
.metric .sub code{
  display:block; max-width:100%; white-space:normal; overflow-wrap:anywhere; word-break:break-word;
  background:#f1f5f9; padding:4px 6px; border-radius:6px; font-size:.9rem; line-height:1.25;
}
/* Stats layout: Equity full width on top, three cards below */
.grid-stats{display:grid;gap:12px}
.row-3{display:grid;gap:12px;grid-template-columns:1fr}
@media (min-width: 960px){ .row-3{grid-template-columns:repeat(3,1fr)} }
/* Equity height */
.card .chart-lg{height:350px}
a {text-decoration: none;}
</style>
</head>
<body>
<div class="wrap">
  <div class="topbar">
    <div class="header">
       <a href="https://ea-marketplace.com" target="_blank" rel="noopener">
        <div class="brand">
        <h1 class="title">&nbsp;&nbsp;TradeJourney</h1>
        <div class="tag">&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Quick notes. Clear insights.</div>
      </div>
      </a> 
	  
    </div>
    <a class="logout" href="?logout=1" title="Sign out">Logout</a>
  </div>

  <div class="tabs">
    <a class="tab <?php echo ($tab==='input'?'active':''); ?>" href="?tab=input">Input</a>
    <a class="tab <?php echo ($tab==='stats'?'active':''); ?>" href="?tab=stats">Statistics</a>
    <a class="tab <?php echo ($tab==='calc'?'active':''); ?>" href="?tab=calc">Calculations</a>
  </div>

  <?php if ($tab==='input') { ?>
    <div class="card">
      <?php if(!empty($msg)){ ?><div class="bad" style="margin-bottom:8px"><?php echo s($msg); ?></div><?php } ?>
      <form method="post">
        <input type="hidden" name="act" value="add">
        <div class="grid">
          <label>Open time <input type="datetime-local" name="ts_open" step="60"></label>
          <label>Close time <input type="datetime-local" name="ts_close" step="60" value="<?php echo s(date('Y-m-d\TH:i')); ?>"></label>
          <label>Symbol <input name="symbol" placeholder="e.g. DE40, XAUUSD" required></label>
          <label>Direction
            <select name="direction"><option>Long</option><option>Short</option></select>
          </label>
          <label>Entry <input name="entry" placeholder="e.g. 15890"></label>
          <label>Exit <input name="exit" placeholder="e.g. 15955"></label>
          <label>SL <input name="sl" placeholder="optional"></label>
          <label>TP <input name="tp" placeholder="optional"></label>
          <label>Volume <input name="volume" value="1"></label>
          <label>Profit (€) <input name="profit_eur" placeholder="enter manually"></label>
          <label>Setup <input name="setup" placeholder="e.g. ORB, Pullback"></label>
          <label style="grid-column:1/-1">Notes <textarea name="comment" rows="2" placeholder="Notes …"></textarea></label>
        </div>
        <div style="margin-top:10px">
          <button class="btn" type="submit">Save</button>
          <span class="small"></span>
        </div>
      </form>
    </div>
	<br>
    <div class="card">
      <h3>Entries</h3>
      <table id="tbl">
        <thead>
          <tr>
            <th>Open</th><th>Open price</th><th>Close</th><th>Close price</th>
            <th>Symbol</th><th>Side</th><th>Profit €</th><th>Setup</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach(array_values($DATA) as $i=>$r){ ?>
            <tr class="clickable" data-idx="<?php echo $i; ?>">
              <td><?php echo s($r['ts_open']); ?></td>
              <td><?php echo s($r['entry']); ?></td>
              <td><?php echo s($r['ts_close']); ?></td>
              <td><?php echo s($r['exit']); ?></td>
              <td><?php echo s($r['symbol']); ?></td>
              <td><?php echo s($r['direction']); ?></td>
              <td class="<?php echo (num($r['profit_eur'])>=0?'ok':'bad'); ?>"><?php echo fmt($r['profit_eur']); ?></td>
              <td><?php echo s($r['setup']); ?></td>
            </tr>
          <?php } ?>
        </tbody>
      </table>
      <div class="small">Hint: click a row to open the edit window.</div>
    </div>
<?php } elseif ($tab==='stats') { ?>
  <div class="grid-stats">
    <!-- TOP ROW: Equity full width -->
    <div class="card">
      <canvas id="equity" class="chart-lg"></canvas>
    </div>

    <!-- SECOND ROW: 3 cards side by side -->
    <div class="row-3">
      <div class="card">
        <h3>KPIs</h3>
        <div class="kpi"><span class="lbl">Trades</span><span class="val"><?php echo $total; ?></span></div>
        <div class="kpi"><span class="lbl">Win / Loss</span><span class="val"><?php echo $wins; ?> / <?php echo $loss; ?></span></div>
        <div class="kpi"><span class="lbl">Hit rate</span><span class="val"><?php echo fmt($hit,2); ?>%</span></div>
        <div class="kpi"><span class="lbl">Gross win / loss</span>
          <span class="val ok"><?php echo fmt($brut_win); ?></span> /
          <span class="val bad"><?php echo fmt($brut_loss); ?></span>
        </div>
        <div class="kpi"><span class="lbl">Profit factor</span><span class="val acc"><?php echo is_finite($pf)?fmt($pf,3):'∞'; ?></span></div>
        <div class="kpi"><span class="lbl">Net PnL</span><span class="val <?php echo ($net>=0?'ok':'bad'); ?>"><?php echo fmt($net); ?></span></div>
        <div class="kpi"><span class="lbl">Avg per trade</span><span class="val"><?php echo fmt($avg); ?></span></div>
      </div>

      <div class="card">
        <canvas id="wl" height="220"></canvas>
      </div>

      <div class="card">
        <canvas id="ls" height="220"></canvas>
      </div>
    </div>
  </div>
<?php } else { /* calc */ ?>

    <div class="metrics">
      <div class="metric"><h4>Total trades</h4><div class="big"><?php echo $total; ?></div><div class="sub">All recorded closes</div></div>
      <div class="metric"><h4>Winners / Losers</h4><div class="big"><?php echo $wins; ?> / <?php echo $loss; ?></div><div class="sub">Hit rate: <?php echo fmt($hit,2); ?>%</div></div>
      <div class="metric"><h4>Gross profit</h4><div class="big ok"><?php echo fmt($brut_win); ?> €</div><div class="sub">Sum of positive trades</div></div>
      <div class="metric"><h4>Gross loss</h4><div class="big bad"><?php echo fmt($brut_loss); ?> €</div><div class="sub">Sum of negative trades</div></div>
      <div class="metric"><h4>Profit factor</h4><div class="big acc"><?php echo is_finite($pf)?fmt($pf,3):'∞'; ?></div><div class="sub">Gross profit / gross loss</div></div>
      <div class="metric"><h4>Net PnL</h4><div class="big <?php echo ($net>=0?'ok':'bad'); ?>"><?php echo fmt($net); ?> €</div><div class="sub">Profit – loss</div></div>
      <div class="metric"><h4>Average per trade</h4><div class="big"><?php echo fmt($avg); ?> €</div><div class="sub">Net PnL / trades</div></div>
      <div class="metric"><h4>CSV file</h4><div class="sub"><?php echo s($CSV_FILE); ?></div></div>
    </div>
  <?php } ?>
</div>

<?php if ($tab!=='input') { ?>
<script>
const labels = <?php echo json_encode($labels, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE); ?>;
const equity = <?php echo json_encode($equity); ?>;
new Chart(document.getElementById('equity'), {
  type:'line',
  data:{ labels:labels, datasets:[{ label:'Equity (€)', data:equity, tension:.25, borderWidth:3, pointRadius:0 }] },
  options:{
    maintainAspectRatio:false,
    plugins:{ legend:{ display:false } },
    scales:{ x:{ display:false } }
  }
});

const wlLabels = <?php echo json_encode($wl_labels); ?>;
const wlWins   = <?php echo json_encode($wl_w); ?>;
const wlLoss   = <?php echo json_encode($wl_l); ?>;
new Chart(document.getElementById('wl'), {
  type:'bar',
  data:{labels:wlLabels, datasets:[
    {label:'Wins', data:wlWins, stack:'a'},
    {label:'Loss', data:wlLoss, stack:'a'}
  ]},
  options:{scales:{y:{stacked:true}, x:{stacked:true}}, plugins:{legend:{position:'top'}}}
});
const lsData = <?php echo json_encode([round($sumL,2), round($sumS,2)]); ?>;
new Chart(document.getElementById('ls'), {
  type:'doughnut',
  data:{labels:['Long PnL','Short PnL'], datasets:[{data:lsData}]},
  options:{plugins:{legend:{position:'bottom'}}}
});
</script>
<?php } else { ?>
<script>
// Row click → centered popup editor
document.querySelectorAll('#tbl tbody tr.clickable').forEach(function(tr){
  tr.addEventListener('click', function(){
    var idx = tr.getAttribute('data-idx');
    const w = 820, h = 780;
    const left = Math.max(0, Math.floor((window.screen.width  - w)/2));
    const top  = Math.max(0, Math.floor((window.screen.height - h)/2));
    const feat = `width=${w},height=${h},left=${left},top=${top},resizable=yes,scrollbars=yes,noopener=yes`;
    window.open('<?php echo basename($_SERVER['PHP_SELF']); ?>?edit='+idx,'edit',feat);
  });
});
</script>
<?php } ?>
</body>
</html>
